'use client'

import { useState, useEffect } from 'react'
import { Card, CardContent } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Badge } from '@/components/ui/badge'
import { 
  Car, 
  Search, 
  Filter, 
  Calendar,
  Users,
  Fuel,
  Settings,
  ChevronRight,
  MapPin
} from 'lucide-react'
import Link from 'next/link'
import ChatSupport from '@/components/ChatSupport'

export default function CarsPage() {
  const [cars, setCars] = useState([])
  const [filteredCars, setFilteredCars] = useState([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState('')
  const [filters, setFilters] = useState({
    brand: '',
    transmission: '',
    fuelType: '',
    minPrice: '',
    maxPrice: ''
  })
  const [showFilters, setShowFilters] = useState(false)

  useEffect(() => {
    fetchCars()
  }, [])

  useEffect(() => {
    filterCars()
  }, [cars, searchTerm, filters])

  const fetchCars = async () => {
    try {
      const response = await fetch('/api/cars')
      const data = await response.json()
      if (data.success) {
        setCars(data.data)
        setFilteredCars(data.data)
      }
    } catch (error) {
      console.error('Failed to fetch cars:', error)
    } finally {
      setLoading(false)
    }
  }

  const filterCars = () => {
    let filtered = cars

    // Search filter
    if (searchTerm) {
      filtered = filtered.filter(car =>
        car.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        car.brand.toLowerCase().includes(searchTerm.toLowerCase()) ||
        car.model.toLowerCase().includes(searchTerm.toLowerCase())
      )
    }

    // Brand filter
    if (filters.brand) {
      filtered = filtered.filter(car => 
        car.brand.toLowerCase().includes(filters.brand.toLowerCase())
      )
    }

    // Transmission filter
    if (filters.transmission) {
      filtered = filtered.filter(car => car.transmission === filters.transmission)
    }

    // Fuel type filter
    if (filters.fuelType) {
      filtered = filtered.filter(car => car.fuelType === filters.fuelType)
    }

    // Price range filter
    if (filters.minPrice) {
      filtered = filtered.filter(car => car.price >= parseFloat(filters.minPrice))
    }
    if (filters.maxPrice) {
      filtered = filtered.filter(car => car.price <= parseFloat(filters.maxPrice))
    }

    setFilteredCars(filtered)
  }

  const clearFilters = () => {
    setFilters({
      brand: '',
      transmission: '',
      fuelType: '',
      minPrice: '',
      maxPrice: ''
    })
    setSearchTerm('')
  }

  const getUniqueBrands = () => {
    return [...new Set(cars.map(car => car.brand))].sort()
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b sticky top-0 z-40">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <Link href="/" className="flex items-center space-x-2">
              <Car className="h-8 w-8 text-blue-600" />
              <span className="text-xl font-bold text-gray-900">Rental Mobil</span>
            </Link>
            <Link href="/">
              <Button variant="outline">Kembali ke Beranda</Button>
            </Link>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Page Title */}
        <div className="text-center mb-8">
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Katalog Armada Kami
          </h1>
          <p className="text-xl text-gray-600 max-w-2xl mx-auto">
            Pilih mobil yang sesuai dengan kebutuhan Anda dari berbagai pilihan armada berkualitas
          </p>
        </div>

        {/* Search and Filters */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <div className="flex flex-col lg:flex-row gap-4">
            {/* Search Bar */}
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-5 w-5" />
                <Input
                  placeholder="Cari mobil berdasarkan nama, merek, atau model..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>

            {/* Filter Toggle */}
            <Button
              variant="outline"
              onClick={() => setShowFilters(!showFilters)}
              className="lg:w-auto"
            >
              <Filter className="mr-2 h-4 w-4" />
              Filter
              {showFilters && <span className="ml-2">▲</span>}
            </Button>
          </div>

          {/* Advanced Filters */}
          {showFilters && (
            <div className="mt-6 pt-6 border-t">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Merek
                  </label>
                  <select
                    value={filters.brand}
                    onChange={(e) => setFilters({...filters, brand: e.target.value})}
                    className="w-full p-2 border rounded-md"
                  >
                    <option value="">Semua Merek</option>
                    {getUniqueBrands().map(brand => (
                      <option key={brand} value={brand}>{brand}</option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Transmisi
                  </label>
                  <select
                    value={filters.transmission}
                    onChange={(e) => setFilters({...filters, transmission: e.target.value})}
                    className="w-full p-2 border rounded-md"
                  >
                    <option value="">Semua Transmisi</option>
                    <option value="MANUAL">Manual</option>
                    <option value="AUTOMATIC">Automatic</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Bahan Bakar
                  </label>
                  <select
                    value={filters.fuelType}
                    onChange={(e) => setFilters({...filters, fuelType: e.target.value})}
                    className="w-full p-2 border rounded-md"
                  >
                    <option value="">Semua Tipe</option>
                    <option value="GASOLINE">Bensin</option>
                    <option value="DIESEL">Diesel</option>
                    <option value="HYBRID">Hybrid</option>
                    <option value="ELECTRIC">Electric</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Harga Minimum
                  </label>
                  <Input
                    type="number"
                    placeholder="Rp 0"
                    value={filters.minPrice}
                    onChange={(e) => setFilters({...filters, minPrice: e.target.value})}
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Harga Maximum
                  </label>
                  <Input
                    type="number"
                    placeholder="Rp 1000000"
                    value={filters.maxPrice}
                    onChange={(e) => setFilters({...filters, maxPrice: e.target.value})}
                  />
                </div>
              </div>

              <div className="flex justify-end mt-4">
                <Button variant="outline" onClick={clearFilters}>
                  Reset Filter
                </Button>
              </div>
            </div>
          )}
        </div>

        {/* Results Count */}
        <div className="flex justify-between items-center mb-6">
          <p className="text-gray-600">
            Menampilkan <span className="font-semibold">{filteredCars.length}</span> dari{' '}
            <span className="font-semibold">{cars.length}</span> mobil
          </p>
        </div>

        {/* Cars Grid */}
        {filteredCars.length === 0 ? (
          <div className="text-center py-12">
            <Car className="h-16 w-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">
              Tidak ada mobil yang ditemukan
            </h3>
            <p className="text-gray-600 mb-4">
              Coba ubah filter atau kata kunci pencarian Anda
            </p>
            <Button onClick={clearFilters}>
              Reset Filter
            </Button>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredCars.map((car) => (
              <Card key={car.id} className="overflow-hidden hover:shadow-lg transition">
                <div className="aspect-video bg-gray-100 relative">
                  {car.image ? (
                    <img 
                      src={car.image} 
                      alt={car.name}
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    <div className="w-full h-full flex items-center justify-center">
                      <Car className="h-16 w-16 text-gray-400" />
                    </div>
                  )}
                  <Badge className="absolute top-4 right-4 bg-blue-600">
                    Rp {car.price.toLocaleString('id-ID')}/hari
                  </Badge>
                  {!car.available && (
                    <Badge className="absolute top-4 left-4 bg-red-600">
                      Tidak Tersedia
                    </Badge>
                  )}
                </div>
                <CardContent className="p-6">
                  <h3 className="text-xl font-semibold mb-2">{car.name}</h3>
                  <p className="text-gray-600 mb-4">{car.brand} {car.model}</p>
                  
                  <div className="grid grid-cols-2 gap-4 mb-4 text-sm">
                    <div className="flex items-center text-gray-600">
                      <Calendar className="h-4 w-4 mr-2" />
                      {car.year}
                    </div>
                    <div className="flex items-center text-gray-600">
                      <Users className="h-4 w-4 mr-2" />
                      {car.seats} Seats
                    </div>
                    <div className="flex items-center text-gray-600">
                      <Settings className="h-4 w-4 mr-2" />
                      {car.transmission === 'MANUAL' ? 'Manual' : 'Automatic'}
                    </div>
                    <div className="flex items-center text-gray-600">
                      <Fuel className="h-4 w-4 mr-2" />
                      {car.fuelType === 'GASOLINE' ? 'Bensin' : 
                       car.fuelType === 'DIESEL' ? 'Diesel' :
                       car.fuelType === 'HYBRID' ? 'Hybrid' : 'Electric'}
                    </div>
                  </div>

                  {car.description && (
                    <p className="text-gray-600 text-sm mb-4 line-clamp-2">
                      {car.description}
                    </p>
                  )}

                  <div className="flex gap-2">
                    <Link href={`/cars/${car.id}`} className="flex-1">
                      <Button 
                        className="w-full" 
                        disabled={!car.available}
                      >
                        {car.available ? 'Lihat Detail & Pesan' : 'Tidak Tersedia'}
                      </Button>
                    </Link>
                    <Button variant="outline" size="sm">
                      <MapPin className="h-4 w-4" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>

      {/* Chat Support */}
      <ChatSupport />
    </div>
  )
}