#!/bin/bash

# 🚀 Production Deployment Script untuk Rental Mobil
# Usage: ./deploy.sh [environment]
# Example: ./deploy.sh vercel

set -e  # Exit on error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${GREEN}🚀 Starting Rental Mobil Deployment...${NC}"

# Configuration
ENVIRONMENT=${1:-production}
BUILD_DIR=".next"
BACKUP_DIR="backups"
TIMESTAMP=$(date +%Y%m%d_%H%M%S)

echo -e "${BLUE}Environment: $ENVIRONMENT${NC}"

# 1. Pre-deployment checks
echo -e "${YELLOW}📋 Running pre-deployment checks...${NC}"

# Check if node_modules exists
if [ ! -d "node_modules" ]; then
    echo -e "${RED}❌ node_modules not found. Running npm install...${NC}"
    npm install
fi

# Check if .env file exists
if [ ! -f ".env.$ENVIRONMENT" ]; then
    echo -e "${RED}❌ .env.$ENVIRONMENT file not found. Please create it first.${NC}"
    echo -e "${YELLOW}📝 Creating .env.$ENVIRONMENT from template...${NC}"
    cp .env.production.example .env.$ENVIRONMENT
    echo -e "${RED}⚠️  Please edit .env.$ENVIRONMENT with your values!${NC}"
    exit 1
fi

echo -e "${GREEN}✅ Pre-deployment checks passed${NC}"

# 2. Create backup
echo -e "${YELLOW}💾 Creating backup...${NC}"
mkdir -p $BACKUP_DIR

# Backup database if exists
if [ -f "db/prod.db" ]; then
    cp "db/prod.db" "$BACKUP_DIR/prod_$TIMESTAMP.db"
    echo -e "${GREEN}✅ Database backed up${NC}"
fi

# 3. Clean previous build
echo -e "${YELLOW}🧹 Cleaning previous build...${NC}"
rm -rf $BUILD_DIR
echo -e "${GREEN}✅ Previous build cleaned${NC}"

# 4. Install dependencies
echo -e "${YELLOW}📦 Installing dependencies...${NC}"
npm ci --only=production
echo -e "${GREEN}✅ Dependencies installed${NC}"

# 5. Generate Prisma client
echo -e "${YELLOW}🗄️ Generating Prisma client...${NC}"
npx prisma generate
echo -e "${GREEN}✅ Prisma client generated${NC}"

# 6. Build application
echo -e "${YELLOW}🏗️ Building application for $ENVIRONMENT...${NC}"
NODE_ENV=$ENVIRONMENT npm run build

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✅ Build completed successfully${NC}"
else
    echo -e "${RED}❌ Build failed${NC}"
    exit 1
fi

# 7. Create production package
echo -e "${YELLOW}📦 Creating production package...${NC}"
cd $BUILD_DIR
npm install --production
cd ..

# 8. Size optimization
echo -e "${YELLOW}📊 Optimizing build size...${NC}"
du -sh $BUILD_DIR
echo -e "${GREEN}Build size: $(du -sh $BUILD_DIR | cut -f1)${NC}"

# 9. Environment-specific deployment
echo -e "${YELLOW}🚀 Deploying to $ENVIRONMENT...${NC}"

case $ENVIRONMENT in
    "vercel")
        echo -e "${BLUE}📡 Deploying to Vercel...${NC}"
        vercel --prod
        ;;
    "netlify")
        echo -e "${BLUE}📡 Deploying to Netlify...${NC}"
        netlify deploy --prod --dir=$BUILD_DIR
        ;;
    "cpanel")
        echo -e "${BLUE}📡 Preparing for cPanel deployment...${NC}"
        echo -e "${GREEN}📁 Build files ready at: $BUILD_DIR${NC}"
        echo -e "${YELLOW}📤 Don't forget to: ${NC}"
        echo -e "${YELLOW}   1. Set NODE_ENV=production${NC}"
        echo -e "${YELLOW}   2. Configure DATABASE_URL${NC}"
        echo -e "${YELLOW}   3. Set document root to $BUILD_DIR${NC}"
        echo -e "${YELLOW}   4. Upload via File Manager or FTP${NC}"
        ;;
    "vps")
        SERVER=${2:-"user@your-server.com"}
        SERVER_PATH=${3:-"/var/www/rental-mobil"}
        
        echo -e "${BLUE}📡 Deploying to VPS: $SERVER${NC}"
        
        # Upload files
        rsync -avz --delete $BUILD_DIR/ $SERVER:$SERVER_PATH/
        
        # Restart application
        ssh $SERVER "cd $SERVER_PATH && pm2 restart rental-mobil"
        
        echo -e "${GREEN}✅ VPS deployment completed${NC}"
        ;;
    *)
        echo -e "${GREEN}✅ Build completed for local deployment${NC}"
        echo -e "${YELLOW}📁 Build files ready at: $BUILD_DIR${NC}"
        echo -e "${BLUE}💡 To deploy to Vercel: ./deploy.sh vercel${NC}"
        ;;
esac

# 10. Post-deployment
echo -e "${GREEN}🎉 Deployment completed successfully!${NC}"
echo -e "${YELLOW}📊 Build statistics:${NC}"
echo -e "${YELLOW}   - Build size: $(du -sh $BUILD_DIR | cut -f1)${NC}"
echo -e "${YELLOW}   - Timestamp: $TIMESTAMP${NC}"

if [ -f "$BACKUP_DIR/prod_$TIMESTAMP.db" ]; then
    echo -e "${YELLOW}   - Database backup: $BACKUP_DIR/prod_$TIMESTAMP.db${NC}"
fi

echo -e "${GREEN}🌐 Website ready for production!${NC}"

# 11. Performance test (optional)
if [ "$ENVIRONMENT" = "vercel" ]; then
    read -p "Do you want to run performance tests? (y/n): " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        echo -e "${YELLOW}🔍 Running performance tests...${NC}"
        
        # Test if production server is running
        if command -v curl &> /dev/null; then
            echo -e "${YELLOW}🌐 Testing website accessibility...${NC}"
            # Check Vercel deployment URL
            sleep 5
            curl -I -s -o /dev/null -w "%{http_code}" https://rental-mobil-website.vercel.app || echo "Server not responding"
        fi
        
        # Check build optimization
        echo -e "${YELLOW}📈 Analyzing bundle size...${NC}"
        if command -v npx &> /dev/null; then
            npx bundlephobia analyze $BUILD_DIR/_next/static/chunks/ 2>/dev/null || echo "Bundle analyzer not available"
        fi
    fi
fi

echo -e "${GREEN}✨ All done! Your rental car website is ready!${NC}"
echo -e "${BLUE}📋 Next steps:${NC}"
echo -e "${YELLOW}   1. Configure custom domain in Vercel dashboard${NC}"
echo -e "${YELLOW}   2. Update DNS settings at your domain provider${NC}"
echo -e "${YELLOW}   3. Set environment variables in Vercel dashboard${NC}"
echo -e "${YELLOW}   4. Test all functionality${NC}"