# 🖥️ Panduan Lengkap Install & Deploy dari Windows

## 📋 Prasyarat Sistem

### Windows Requirements:
- Windows 10/11 (64-bit)
- Git for Windows
- Node.js 18+ 
- Browser modern (Chrome, Firefox, Edge)

---

## 🔧 Tahap 1: Install Software di Windows

### A. Install Git for Windows
**Option 1: Git SCM (Recommended)**
```bash
# Download dari https://git-scm.com/download/win
# Install dengan default settings
```

**Option 2: GitHub Desktop**
```bash
# Download dari https://desktop.github.com
# Install dan login dengan GitHub
```

**Option 3: Git for Windows via WSL**
```bash
# Install WSL2: wsl --install -d Ubuntu
# Install Git di WSL: sudo apt install git
```

### B. Install Node.js
**Option 1: Official Installer (Recommended)**
```bash
# Download dari https://nodejs.org/en/download/
# Pilih Windows Installer (.msi)
# Install dengan "Add to PATH"
```

**Option 2: Chocolatey**
```bash
# Install Chocolatey: powershell -Command "iwr -useb https://chocolatey.org/install.ps1 | iex"
# Install Node.js: choco install nodejs
```

**Option 3: WSL**
```bash
# Di WSL Ubuntu: curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs
```

### C. Install VS Code (Recommended)
```bash
# Download dari https://code.visualstudio.com/
# Install dengan extensions:
# - ES7+ React/React/Redux snippets
# - Prettier
# - GitLens
```

---

## 🔧 Tahap 2: Setup Project

### A. Buka Command Prompt/PowerShell
```powershell
# Buka Command Prompt sebagai Administrator
# Atau PowerShell (lebih recommended)
```

### B. Clone Repository
```bash
# Via HTTPS (jika repo private)
git clone https://github.com/username/rental-mobil.git

# Via SSH (jika punya akses)
git clone git@github.com:username/rental-mobil.git

# Via GitHub Desktop
# File → Clone Repository → masukkan URL
```

### C. Masuk ke Folder Project
```bash
cd rental-mobil
```

### D. Install Dependencies
```bash
# Install dependencies
npm install

# Atau dengan yarn
yarn install
```

---

## 🔧 Tahap 3: Setup Environment

### A. Buat Environment File
```bash
# Copy template
copy .env.production.example .env.local

# Edit dengan VS Code
code .env.local
```

### B. Isi Environment Variables
```env
# Development
DATABASE_URL="file:./db/dev.db"
NEXTAUTH_URL="http://localhost:3000"
NEXTAUTH_SECRET="your-secret-key-for-dev"
NODE_ENV="development"

# Production (nanti)
DATABASE_URL="file:./db/prod.db"
NEXTAUTH_URL="https://yourdomain.com"
NEXTAUTH_SECRET="your-super-secret-key-min-32-characters"
NODE_ENV="production"
```

---

## 🔧 Tahap 4: Test Lokal

### A. Jalankan Development Server
```bash
# Start development server
npm run dev

# Atau dengan PowerShell
npm start
```

### B. Test di Browser
```bash
# Buka browser
http://localhost:3000

# Test semua fitur:
# - Homepage
# - Katalog mobil
# - Detail mobil
# - Form booking
# - Admin dashboard
# - Chat support
```

---

## 🌐 Tahap 5: Setup Vercel via Browser

### A. Install Vercel CLI
```bash
# Install Vercel CLI globally
npm install -g vercel

# Verify installation
vercel --version
```

### B. Login ke Vercel
```bash
# Login ke Vercel
vercel login

# Akan membuka browser untuk authentication
# Follow instruksi di browser
```

### C. Setup Project untuk Vercel
```bash
# Di folder project
vercel link

# Atau deploy langsung
vercel --prod
```

---

## 🚀 Tahap 6: Deploy via Browser Vercel

### A. Buka Vercel Dashboard
1. Buka [vercel.com](https://vercel.com)
2. Login dengan akun Anda
3. Dashboard akan terbuka

### B. Deploy Pertama
1. Klik **"Add New..."** atau **"Import Project"**
2. Pilih **"Git Repository"**
3. Masukkan repository URL: `https://github.com/username/rental-mobil.git`
4. Klik **"Deploy"**

### B. Konfigurasi Project Settings
1. Di project dashboard, klik **"Settings"**
2. **Environment Variables**:
   ```
   DATABASE_URL=file:./db/prod.db
   NEXTAUTH_SECRET=your-secret-key
   NEXTAUTH_URL=https://yourdomain.com
   NODE_ENV=production
   ```
3. **Build Settings**:
   - Build Command: `npm run build`
   - Output Directory: `.next`
   - Node.js Version: `18.x`

### C. Setup Custom Domain
1. Klik **"Domains"** tab
2. Klik **"Add Domain"**
3. Masukkan domain Anda: `yourdomain.com`
4. Pilih **"Production Environment"**
5. Klik **"Add"**

### D. Konfigurasi DNS
Di domain provider Anda, tambahkan:
```
Type: CNAME
Name: @ (atau www)
Value: cname.vercel-dns.com
TTL: 300
```

---

## 🔄 Tahap 7: Update Code untuk Production

### A. Update Konfigurasi Next.js
```javascript
// next.config.js
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'standalone',
  experimental: {
    optimizeCss: true,
  },
  images: {
    domains: ['yourdomain.com', 'www.yourdomain.com'],
    formats: ['image/webp', 'image/avif']
  },
  compress: true,
}

module.exports = nextConfig
```

### B. Update Layout untuk Custom Domain
```javascript
// app/layout.tsx
export const metadata = {
  title: 'Rental Mobil Professional',
  metadataBase: new URL('https://yourdomain.com'),
  openGraph: {
    url: 'https://yourdomain.com',
    siteName: 'Rental Mobil',
  },
}
```

---

## 📊 Tahap 8: Monitoring & Debugging

### A. Vercel Dashboard Monitoring
1. **Analytics Tab**:
   - Page views
   - Unique visitors
   - Performance metrics
   - Geographic distribution

2. **Logs Tab**:
   - Real-time error tracking
   - Function logs
   - Build logs

3. **Functions Tab**:
   - API performance
   - Function execution time
   - Error rates

### B. Debugging Tools
```bash
# Vercel CLI logs
vercel logs

# Real-time logs
vercel logs --follow

# Specific function logs
vercel logs --filter=api/cars
```

---

## 🔧 Tahap 9: Advanced Windows Setup

### A. Windows Terminal Setup
**PowerShell 7+ (Recommended)**
```powershell
# Install PowerShell modules
Install-Module -Name Terminal-Icons
Install-Module -Name PSReadLine

# Custom profile
notepad $PROFILE
```

**Windows Terminal (Modern)**
```bash
# Install dari Microsoft Store
# Atau gunakan Windows Terminal Preview
```

### B. Git Configuration Windows
```bash
# Setup Git credential manager
git config --global credential.helper manager

# Setup default editor
git config --global core.editor "code --wait"

# Setup line endings
git config --global core.autocrlf true
git config --global core.eol lf
```

### C. Environment Variables Windows
```powershell
# PowerShell
$env:NODE_ENV="production"

# Command Prompt
set NODE_ENV=production

# Di .env file
NODE_ENV=production
```

---

## 🛠️ Tahap 10: Troubleshooting Windows

### A. Common Issues

#### 1. Port 3000 Already in Use
```bash
# Cek proses yang menggunakan port 3000
netstat -ano | findstr :3000

# Kill proses
taskkill /F /IM node.exe

# Atau gunakan port berbeda
PORT=3001 npm run dev
```

#### 2. Permission Issues
```bash
# Run sebagai Administrator
# Buka PowerShell sebagai Administrator

# Atau setup file permissions
icacls . /grant Users:(OI)(CI)F
```

#### 3. Node Modules Issues
```bash
# Clear cache
npm cache clean --force

# Delete node_modules
rm -rf node_modules

# Reinstall
npm install
```

#### 4. Environment Variables Not Loading
```bash
# Restart PowerShell
$env:NODE_ENV="production"

# Atau restart command prompt
set NODE_ENV=production
```

### B. Performance Optimization Windows
```bash
# Windows Defender exclusion
powershell -Command "Add-MpPreference -ExclusionPath '$PWD\node_modules'"

# Optimize untuk Windows
npm run build --platform=win32
```

---

## 📋 Script Windows Otomatis

### A. Deploy Script untuk Windows
```batch
@echo off
title Rental Mobil Deployment
color 0A

echo.
echo ====================================
echo    RENTAL MOBIL DEPLOYMENT
echo ====================================
echo.

:: Check prerequisites
echo [1] Checking Git...
git --version >nul 2>&1
if %errorlevel% neq 0 (
    echo [ERROR] Git not found. Please install Git first.
    pause
    exit /b 1
)

echo [✓] Git found
echo [2] Checking Node.js...
node --version >nul 2>&1
if %errorlevel% neq 0 (
    echo [ERROR] Node.js not found. Please install Node.js first.
    pause
    exit /b 1
)

echo [✓] Node.js found
echo [3] Installing dependencies...
npm ci >nul 2>&1
if %errorlevel% neq 0 (
    echo [ERROR] Failed to install dependencies.
    pause
    exit /b 1
)

echo [✓] Dependencies installed
echo [4] Building project...
npm run build >nul 2>&1
if %errorlevel% neq 0 (
    echo [ERROR] Build failed.
    pause
    exit /b 1
)

echo [✓] Build completed
echo [5] Opening Vercel...
start https://vercel.com/dashboard

echo.
echo ====================================
echo    DEPLOYMENT READY!
echo ====================================
echo.
echo Next steps:
echo 1. Login to Vercel
echo 2. Import/add your project
echo 3. Configure environment variables
echo 4. Deploy to production
echo.
pause
```

Simpan sebagai `deploy.bat` dan jalankan dengan double-click.

---

## 🌐 Vercel Web App Setup

### A. Install Vercel Web App
1. Buka [vercel.com](https://vercel.com)
2. Download **Vercel Desktop App**
3. Install dan login

### B. Deploy via Web App
1. Buka Vercel Desktop App
2. **"Add New..."** → **"Import Project"**
3. Drag & drop folder project
4. Klik **"Deploy"**
5. Configure environment variables di web app

---

## 📱 Mobile Development (Optional)

### A. React Native Development
```bash
# Install Expo CLI
npm install -g @expo/cli

# Initialize Expo
expo init

# Run di mobile device
expo start
```

### B. Progressive Web App
```javascript
// Tambahkan ke next.config.js
const nextConfig = {
  experimental: {
    appDir: true,
  },
  pwa: {
    dest: 'public',
    register: true,
    skipWaiting: true,
  },
}
```

---

## 🔐 Security Best Practices Windows

### A. Windows Security
```bash
# Enable Windows Defender real-time protection
powershell -Command "Set-MpPreference -DisableRealtimeMonitoring $false"

# Firewall configuration
netsh advfirewall set allprofiles state on
netsh advfirewall firewall add rule name="Node.js" dir=in action=allow program="node.exe"
```

### B. Credential Management
```bash
# Use Windows Credential Manager
git config --global credential.helper manager

# Atau environment variables
$env:NODE_ENV="production"
```

---

## 📊 Testing & Quality Assurance

### A. Browser Testing
```bash
# Test di berbagai browser
# - Chrome (Chromium)
# - Firefox
# - Microsoft Edge
# - Safari (jika ada)
```

### B. Performance Testing
```bash
# Install Lighthouse CLI
npm install -g lighthouse

# Run performance audit
lighthouse http://localhost:3000 --view --preset=desktop
```

### C. Cross-browser Testing
```bash
# Test responsive design
# Test mobile viewports
# Test tablet layouts
# Test desktop layouts
```

---

## 🎯 Tips & Best Practices

### A. Development Workflow
1. **Gunakan Git branches**:
   - `main` untuk production
   - `develop` untuk development
   - `feature/` untuk fitur baru

2. **Commit messages yang jelas**:
   ```
   feat: tambah fitur booking
   fix: perbaiki mobile responsiveness
   docs: update panduan deployment
   ```

3. **Regular testing**:
   - Test setiap perubahan
   - Run automated tests
   - Manual testing di berbagai browser

### B. Performance Optimization
1. **Image optimization**:
   - Gunakan format WebP/AVIF
   - Compress images
   - Lazy loading untuk gambar

2. **Code splitting**:
   - Dynamic imports
   - Route-based code splitting
   - Component lazy loading

3. **Database optimization**:
   - Gunakan indexes
   - Query optimization
   - Connection pooling

---

## 📞 Support & Resources

### A. Documentation
- [Next.js Docs](https://nextjs.org/docs)
- [Vercel Docs](https://vercel.com/docs)
- [React Docs](https://react.dev)

### B. Community Support
- [Stack Overflow](https://stackoverflow.com)
- [GitHub Issues](https://github.com/vercel/vercel/issues)
- [Discord Vercel](https://discord.gg/vercel)

### C. Help Commands
```bash
# Vercel help
vercel --help

# Git help
git --help

# NPM help
npm --help
```

---

## 🎉 Deployment Success Checklist

### Pre-Deployment:
- [ ] Git repository siap
- [ ] Dependencies terinstall
- [ ] Environment variables dikonfigurasi
- [ ] Build berhasil tanpa error
- [ ] Local testing selesai

### Post-Deployment:
- [ ] Website accessible via custom domain
- [ ] HTTPS working dengan green padlock
- [ ] All pages loading (200 status)
- [ ] Forms dan API berfungsi
- [ ] Mobile responsive design
- [ ] No console errors
- [ ] Performance score acceptable (>90)
- [ ] SEO meta tags correct
- [ ] SSL certificate valid
- [ ] Monitoring aktif

---

## 🚀 Quick Start Summary

Untuk pemula Windows:

1. **Install**: Git + Node.js + VS Code
2. **Clone**: Repository rental mobil
3. **Setup**: Environment variables
4. **Test**: Jalankan `npm run dev`
5. **Deploy**: Via Vercel dashboard atau CLI
6. **Monitor**: Via Vercel analytics

Website rental mobil Anda siap untuk production dari Windows! 🎉