# 🚀 Quick Start Vercel Deployment Guide

## 📋 Langkah 1: Persiapan Awal

### A. Install Vercel CLI
```bash
npm install -g vercel
```

### B. Login ke Vercel
```bash
vercel login
```

## 🔧 Langkah 2: Deploy Pertama

### A. Deploy Project
```bash
# Build dan deploy ke Vercel
npm run build
vercel --prod
```

### B. Setup Project di Vercel
1. Buka [Vercel Dashboard](https://vercel.com/dashboard)
2. Project akan otomatis terdeteksi
3. Catat project URL (misal: `rental-mobil-website.vercel.app`)

## 🌐 Langkah 3: Setup Custom Domain

### A. Tambah Domain di Vercel
1. Di Vercel Dashboard, pilih project Anda
2. Klik tab **"Domains"**
3. Klik **"Add Domain"**
4. Masukkan domain Anda: `yourdomain.com`
5. Klik **"Add"**

### B. Konfigurasi DNS
Di domain provider Anda, tambahkan DNS record:

```
Type: CNAME
Name: @ (atau www)
Value: rental-mobil-website.vercel.app
TTL: 300
```

**⏱️ Tunggu propagasi DNS (1-24 jam)**

## 🔧 Langkah 4: Environment Variables

### A. Setup di Vercel Dashboard
1. Di project settings, klik **"Environment Variables"**
2. Tambahkan variables berikut:

```env
DATABASE_URL=file:./db/prod.db
NEXTAUTH_SECRET=your-super-secret-key-min-32-characters
NEXTAUTH_URL=https://yourdomain.com
NODE_ENV=production
```

### B. Generate NEXTAUTH_SECRET
```bash
# Generate secure random string
node -e "console.log(require('crypto').randomBytes(32).toString('hex'))"
```

## 🚀 Langkah 5: Deploy Final

### A. Update Code untuk Custom Domain
Update file berikut untuk mendukung custom domain:

#### 1. Update next.config.js
```javascript
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'standalone',
  images: {
    domains: ['yourdomain.com', 'www.yourdomain.com'],
  },
}

module.exports = nextConfig
```

#### 2. Update layout.tsx
```javascript
// app/layout.tsx
export const metadata = {
  title: 'Rental Mobil Professional',
  metadataBase: new URL('https://yourdomain.com'),
  openGraph: {
    url: 'https://yourdomain.com',
  },
}
```

### B. Deploy dengan Environment Variables
```bash
# Deploy dengan semua environment variables
vercel --prod --env DATABASE_URL,NEXTAUTH_SECRET,NEXTAUTH_URL
```

## ✅ Langkah 6: Testing

### A. Test Custom Domain
```bash
# Test website
curl -I https://yourdomain.com

# Test API endpoints
curl https://yourdomain.com/api/cars
```

### B. Check SSL Certificate
```bash
# Check SSL status
curl -I https://yourdomain.com | grep -i "ssl"
```

## 📊 Langkah 7: Monitoring

### A. Vercel Dashboard
- Monitor **"Analytics"** untuk traffic
- Monitor **"Logs"** untuk error tracking
- Monitor **"Functions"** untuk API performance

### B. Performance Testing
```bash
# Install Lighthouse
npm install -g lighthouse

# Run performance audit
lighthouse https://yourdomain.com --view
```

## 🔧 Troubleshooting

### Common Issues & Solutions:

#### 1. Domain Not Working
```bash
# Check DNS propagation
nslookup yourdomain.com

# Check Vercel domain status
vercel domains ls
```

#### 2. Build Errors
```bash
# Clean build
rm -rf .next node_modules
npm install
npm run build
```

#### 3. Environment Variables Missing
```bash
# Check current variables
vercel env ls

# Add missing variables
vercel env add NEXTAUTH_SECRET
```

## 🎉 Success Checklist

Setelah deployment, pastikan:

- [ ] Website accessible via custom domain
- [ ] HTTPS working dengan SSL certificate
- [ ] All pages loading correctly
- [ ] Forms submitting properly
- [ ] API endpoints responding
- [ ] No console errors
- [ ] Mobile responsive design
- [ ] Performance score acceptable (>90)

## 📞 Support

### Resources:
- [Vercel Documentation](https://vercel.com/docs)
- [Custom Domains Guide](https://vercel.com/docs/concepts/projects/custom-domains)
- [Environment Variables](https://vercel.com/docs/concepts/projects/environment-variables)

### Quick Commands:
```bash
# Deploy update
npm run build && vercel --prod

# Check logs
vercel logs

# Redeploy (force)
vercel --prod --force
```

Website rental mobil Anda siap go-live dengan domain sendiri! 🚀